<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class SwiftCash_Webhook {

    public static function register_routes() {
        register_rest_route( 'swiftcash/v1', '/webhook', array(
            'methods'  => 'POST',
            'callback' => array( __CLASS__, 'handle' ),
            'permission_callback' => '__return_true',
        ) );
    }

    public static function handle( WP_REST_Request $request ) {
        $identifier = sanitize_text_field( $request->get_param( 'identifier' ) );
        $status     = sanitize_text_field( $request->get_param( 'status' ) );
        $signature  = $request->get_param( 'signature' );
        $data_param = $request->get_param( 'data' );

        if ( empty( $identifier ) || empty( $status ) ) {
            return new WP_REST_Response( array( 'ok' => false, 'error' => 'missing_params' ), 400 );
        }

        // Parse 'data' which might be JSON string or array.
        $data = array();
        if ( is_array( $data_param ) ) {
            $data = $data_param;
        } elseif ( is_string( $data_param ) ) {
            $decoded = json_decode( $data_param, true );
            if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
                $data = $decoded;
            }
        }

        $order_id = intval( $identifier );
        $order    = wc_get_order( $order_id );
        if ( ! $order ) {
            return new WP_REST_Response( array( 'ok' => false, 'error' => 'order_not_found' ), 404 );
        }

        // Verify signature with secret: strtoupper( HMAC_SHA256( amount . identifier, secret ) )
        $settings = get_option( 'woocommerce_swiftcash_gateway_settings', array() );
        $secret   = isset( $settings['secret_key'] ) ? $settings['secret_key'] : '';

        if ( ! empty( $secret ) && ! empty( $signature ) && isset( $data['amount'] ) ) {
            $customKey  = (string) $data['amount'] . (string) $identifier;
            $expected   = strtoupper( hash_hmac( 'sha256', $customKey, $secret ) );

            if ( ! hash_equals( $expected, (string) $signature ) ) {
                $order->add_order_note( __( 'SwiftCash IPN: signature mismatch.', 'swiftcash' ) );
                return new WP_REST_Response( array( 'ok' => false, 'error' => 'bad_signature' ), 401 );
            }
        }

        if ( 'success' === strtolower( $status ) ) {
            $order->payment_complete();
            $order->add_order_note( __( 'SwiftCash: payment confirmed via IPN.', 'swiftcash' ) );
        } else {
            $order->update_status( 'failed', __( 'SwiftCash: payment failed via IPN.', 'swiftcash' ) );
        }

        return new WP_REST_Response( array( 'ok' => true ), 200 );
    }
}
