<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry;

if ( class_exists( '\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {

    class WC_Gateway_SwiftCash_Blocks extends AbstractPaymentMethodType {

        protected $name = 'swiftcash_gateway';
        protected $settings = array();

        public function initialize() {
            $this->settings = get_option( 'woocommerce_swiftcash_gateway_settings', array() );

            // Handle Store API checkout for Blocks.
            add_action( 'woocommerce_rest_checkout_process_payment_with_context', array( $this, 'process_payment_with_context' ), 10, 2 );
        }

        public function is_active() {
            $enabled = isset( $this->settings['enabled'] ) ? $this->settings['enabled'] : 'no';
            $pub     = isset( $this->settings['public_key'] ) ? $this->settings['public_key'] : '';
            return ( 'yes' === $enabled && ! empty( $pub ) );
        }

        public function get_payment_method_script_handles() {
            // Register a tiny client script that registers the method in the Checkout block.
            $handle = 'swiftcash-blocks';
            $script_url = SWIFTCASH_PLUGIN_URL . 'assets/js/blocks-swiftcash.js';
            $ver = SWIFTCASH_PLUGIN_VERSION;
            wp_register_script(
                $handle,
                $script_url,
                array( 'wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-i18n' ),
                $ver,
                true
            );
            return array( $handle );
        }

        public function get_payment_method_script_handles_for_admin() {
            return $this->get_payment_method_script_handles();
        }

        public function get_payment_method_data() {
            return array(
                'title'       => isset( $this->settings['title'] ) ? $this->settings['title'] : __( 'SwiftCash', 'swiftcash' ),
                'description' => isset( $this->settings['description'] ) ? $this->settings['description'] : __( 'You will be redirected to SwiftCash to complete payment.', 'swiftcash' ),
                'supports'    => array(),
            );
        }

        /**
         * Store API processing for block checkout.
         */
        public function process_payment_with_context( $context, $result ) {
            if ( $context->payment_method !== 'swiftcash_gateway' ) {
                return;
            }

            $order = $context->order;
            if ( ! $order ) {
                throw new \Exception( __( 'SwiftCash: Order not found.', 'swiftcash' ) );
            }

            $gateways = WC()->payment_gateways()->payment_gateways();
            if ( empty( $gateways['swiftcash_gateway'] ) || ! $gateways['swiftcash_gateway'] instanceof \WC_Gateway_SwiftCash ) {
                throw new \Exception( __( 'SwiftCash: Gateway not available.', 'swiftcash' ) );
            }

            /** @var \WC_Gateway_SwiftCash $gateway */
            $gateway  = $gateways['swiftcash_gateway'];
            $redirect = $gateway->initiate_and_get_redirect_url( $order );

            if ( is_wp_error( $redirect ) ) {
                throw new \Exception( $redirect->get_error_message() );
            }

            $order->update_status( 'pending', __( 'Redirecting to SwiftCash to complete payment.', 'swiftcash' ) );
            $result->set_status( 'success' );
            $result->set_redirect_url( $redirect );
        }
    }
}
