<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class WC_Gateway_SwiftCash extends WC_Payment_Gateway {

    public function __construct() {
        $this->id                 = 'swiftcash_gateway';
        $this->method_title       = __( 'SwiftCash', 'swiftcash' );
        $this->method_description = __( 'Redirect customers to SwiftCash to complete payment.', 'swiftcash' );
        $this->title              = __( 'SwiftCash', 'swiftcash' );
        $this->description        = __( 'You will be redirected to SwiftCash to complete payment.', 'swiftcash' );
        $this->has_fields         = false;
        $this->supports           = array( 'products' );

        $this->init_form_fields();
        $this->init_settings();

        $this->enabled         = $this->get_option( 'enabled', 'no' );
        $this->title           = $this->get_option( 'title', $this->title );
        $this->description     = $this->get_option( 'description', $this->description );
        $this->public_key      = $this->get_option( 'public_key', '' );
        $this->secret_key      = $this->get_option( 'secret_key', '' );
        $this->environment     = $this->get_option( 'environment', 'sandbox' ); // sandbox | live
        $this->site_logo       = $this->get_option( 'site_logo', '' );
        $this->checkout_theme  = $this->get_option( 'checkout_theme', 'light' ); // light | dark
        $this->payload_format  = $this->get_option( 'payload_format', 'form' ); // form | json
        $this->debug           = 'yes' === $this->get_option( 'debug', 'no' );

        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'   => __( 'Enable/Disable', 'swiftcash' ),
                'type'    => 'checkbox',
                'label'   => __( 'Enable SwiftCash', 'swiftcash' ),
                'default' => 'no',
            ),
            'title' => array(
                'title'       => __( 'Title', 'swiftcash' ),
                'type'        => 'text',
                'description' => __( 'Controls the payment method title customers see during checkout.', 'swiftcash' ),
                'default'     => __( 'SwiftCash', 'swiftcash' ),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __( 'Description', 'swiftcash' ),
                'type'        => 'textarea',
                'description' => __( 'Payment method description customers see during checkout.', 'swiftcash' ),
                'default'     => __( 'You will be redirected to SwiftCash to complete payment.', 'swiftcash' ),
                'desc_tip'    => true,
            ),
            'public_key' => array(
                'title'       => __( 'Public Key', 'swiftcash' ),
                'type'        => 'text',
                'description' => __( 'Your SwiftCash public API key (from SwiftCash dashboard).', 'swiftcash' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'secret_key' => array(
                'title'       => __( 'Secret Key', 'swiftcash' ),
                'type'        => 'password',
                'description' => __( 'Your SwiftCash secret key (used to verify IPN signature).', 'swiftcash' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'environment' => array(
                'title'       => __( 'Environment', 'swiftcash' ),
                'type'        => 'select',
                'description' => __( 'Use Sandbox for testing; switch to Live for production.', 'swiftcash' ),
                'default'     => 'sandbox',
                'options'     => array(
                    'sandbox' => __( 'Sandbox', 'swiftcash' ),
                    'live'    => __( 'Live', 'swiftcash' ),
                ),
            ),
            'site_logo' => array(
                'title'       => __( 'Site Logo URL', 'swiftcash' ),
                'type'        => 'text',
                'description' => __( 'Absolute URL to a logo to display on SwiftCash checkout.', 'swiftcash' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'checkout_theme' => array(
                'title'       => __( 'Checkout Theme', 'swiftcash' ),
                'type'        => 'select',
                'description' => __( 'SwiftCash checkout theme.', 'swiftcash' ),
                'default'     => 'light',
                'options'     => array(
                    'light' => __( 'Light', 'swiftcash' ),
                    'dark'  => __( 'Dark', 'swiftcash' ),
                ),
            ),
            'payload_format' => array(
                'title'       => __( 'Payload Format', 'swiftcash' ),
                'type'        => 'select',
                'description' => __( 'How to send the initiate request: URL-encoded form (default) or JSON.', 'swiftcash' ),
                'default'     => 'form',
                'options'     => array(
                    'form' => __( 'Form (application/x-www-form-urlencoded)', 'swiftcash' ),
                    'json' => __( 'JSON (application/json)', 'swiftcash' ),
                ),
            ),
            'debug' => array(
                'title'       => __( 'Debug Log', 'swiftcash' ),
                'type'        => 'checkbox',
                'label'       => __( 'Enable logging to WooCommerce logs', 'swiftcash' ),
                'default'     => 'no',
                'description' => __( 'Logs request/response (with secrets redacted) to WooCommerce > Status > Logs (source: swiftcash).', 'swiftcash' ),
            ),
        );
    }

    public function admin_options() {
        echo '<h2>' . esc_html( $this->get_method_title() ) . '</h2>';
        echo wp_kses_post( wpautop( $this->get_method_description() ) );
        echo '<table class="form-table">';
        $this->generate_settings_html();
        echo '</table>';
        echo '<p>' . esc_html__( 'IPN Webhook URL:', 'swiftcash' ) . ' <code>' . esc_html( rest_url( 'swiftcash/v1/webhook' ) ) . '</code></p>';
    }

    public function is_available() {
        return ( 'yes' === $this->enabled ) && ! empty( $this->public_key );
    }

    protected function maybe_get_logo_url( $provided ) {
        if ( ! empty( $provided ) ) { return $provided; }
        $logo_id = get_theme_mod( 'custom_logo' );
        if ( $logo_id ) {
            $image = wp_get_attachment_image_src( $logo_id, 'full' );
            if ( ! empty( $image[0] ) ) { return $image[0]; }
        }
        $site_icon = get_site_icon_url();
        if ( $site_icon ) { return $site_icon; }
        return '';
    }

    protected function get_endpoint() {
        return ( 'live' === $this->environment )
            ? 'https://swiftcash.app/payment/initiate'
            : 'https://swiftcash.app/sandbox/payment/initiate';
    }

    protected function log( $message, $context = array() ) {
        if ( empty( $this->debug ) ) { return; }
        $logger = wc_get_logger();
        $logger->info( $message . ' ' . wp_json_encode( $context ), array( 'source' => 'swiftcash' ) );
    }

    protected function redact( $arr ) {
        if ( isset( $arr['public_key'] ) ) { $arr['public_key'] = substr( $arr['public_key'], 0, 6 ) . '…'; }
        if ( isset( $arr['secret_key'] ) ) { $arr['secret_key'] = '****'; }
        return $arr;
    }

    protected function build_params( $order ) {
        $logo = $this->maybe_get_logo_url( $this->site_logo );
        return array(
            'public_key'     => (string) $this->public_key,
            'identifier'     => (string) $order->get_id(),
            'currency'       => (string) $order->get_currency(),
            'amount'         => (string) wc_format_decimal( $order->get_total(), 2 ),
            'details'        => sprintf( __( 'WooCommerce Order #%s', 'swiftcash' ), $order->get_order_number() ),
            'ipn_url'        => (string) rest_url( 'swiftcash/v1/webhook' ),
            'success_url'    => (string) $this->get_return_url( $order ),
            'cancel_url'     => (string) wc_get_checkout_url(),
            'site_logo'      => (string) $logo,
            'checkout_theme' => (string) $this->checkout_theme,
            'customer_name'  => (string) trim( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() ),
            'customer_email' => (string) $order->get_billing_email(),
        );
    }

    /**
     * Initiate payment via SwiftCash API and return a redirect URL or WP_Error.
     */
    public function initiate_and_get_redirect_url( $order ) {
        $endpoint = $this->get_endpoint();
        $params   = $this->build_params( $order );
        $args     = array(
            'timeout' => 30,
            'headers' => array( 'Accept' => 'application/json' ),
        );

        if ( 'json' === $this->payload_format ) {
            $args['headers']['Content-Type'] = 'application/json; charset=utf-8';
            $args['body'] = wp_json_encode( $params );
        } else {
            $args['body'] = $params; // form-encoded
        }

        $this->log( 'Initiate request', array( 'endpoint' => $endpoint, 'payload_format' => $this->payload_format, 'body' => $this->redact( $params ) ) );

        $response = wp_remote_post( $endpoint, $args );

        if ( is_wp_error( $response ) ) {
            $this->log( 'HTTP error', array( 'error' => $response->get_error_message() ) );
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $ctype= wp_remote_retrieve_header( $response, 'content-type' );

        $this->log( 'Initiate response', array( 'status' => $code, 'ctype' => $ctype, 'body_snippet' => substr( (string) $body, 0, 400 ) ) );

        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'swiftcash_bad_response', __( 'SwiftCash: Invalid response from payment server.', 'swiftcash' ) );
        }

        $data = json_decode( $body, true );
        if ( null === $data && json_last_error() !== JSON_ERROR_NONE ) {
            return new WP_Error( 'swiftcash_not_json', sprintf( __( 'SwiftCash: Invalid response from payment server. Body starts with: %s', 'swiftcash' ), esc_html( substr( (string) $body, 0, 80 ) ) ) );
        }

        // Interpret various shapes.
        $url = '';
        if ( ! empty( $data['url'] ) ) { $url = $data['url']; }
        elseif ( ! empty( $data['payment_url'] ) ) { $url = $data['payment_url']; }
        elseif ( ! empty( $data['redirect_url'] ) ) { $url = $data['redirect_url']; }
        elseif ( ! empty( $data['data'] ) && is_array( $data['data'] ) ) {
            foreach ( array( 'url', 'payment_url', 'redirect_url' ) as $k ) {
                if ( ! empty( $data['data'][ $k ] ) ) { $url = $data['data'][ $k ]; break; }
            }
        }

        $error_flag = isset( $data['error'] ) ? $data['error'] : null;
        $message    = isset( $data['message'] ) ? $data['message'] : '';

        $is_error = false;
        if ( is_bool( $error_flag ) ) { $is_error = $error_flag; }
        elseif ( is_string( $error_flag ) ) { $is_error = in_array( strtolower( $error_flag ), array( 'true', '1', 'error' ), true ); }
        elseif ( is_numeric( $error_flag ) ) { $is_error = (int) $error_flag === 1; }

        if ( $is_error ) {
            return new WP_Error( 'swiftcash_api_error', 'SwiftCash: ' . ( $message ?: __( 'Payment initiation failed.', 'swiftcash' ) ) );
        }

        if ( ! empty( $url ) ) {
            return esc_url_raw( $url );
        }

        return new WP_Error( 'swiftcash_unexpected', __( 'SwiftCash: Unexpected response from payment server.', 'swiftcash' ) );
    }

    public function process_payment( $order_id ) {
        $order = wc_get_order( $order_id );
        $redirect = $this->initiate_and_get_redirect_url( $order );

        if ( is_wp_error( $redirect ) ) {
            wc_add_notice( $redirect->get_error_message(), 'error' );
            return array( 'result' => 'failure' );
        }

        $order->update_status( 'pending', __( 'Redirecting to SwiftCash to complete payment.', 'swiftcash' ) );

        return array(
            'result'   => 'success',
            'redirect' => $redirect,
        );
    }
}
